
%%%% READ DATA / PRESSURE / ACCELERATION 

%---------------------------------------
csvFiles = dir('*.csv');
fs = 5000;            % [Hz]
windowSize = 10;      % 
cf_Volt_Pa = (88.96 / 0.000126);  % Volt to Pa

allX = [];
allZ = [];
allPressure = [];

%---------------------------------------
% Read .csv
%---------------------------------------
for k = 1:length(csvFiles)
    filename = csvFiles(k).name;
    try
        opts = detectImportOptions(filename, 'NumHeaderLines', 6, ...
                                   'Delimiter', '\t', ...
                                   'VariableNamingRule', 'preserve');

        data = readtable(filename, opts);

    
        hasX = ismember('X (V)', data.Properties.VariableNames);
        hasZ = ismember('Z (V)', data.Properties.VariableNames);
        hasP = ismember('IMPACT SENSOR (V)', data.Properties.VariableNames);

        if hasX && hasZ && hasP
            allX = [allX; data.('X (V)')];
            allZ = [allZ; data.('Z (V)')];
            allPressure = [allPressure; data.('IMPACT SENSOR (V)')];
        else
            fprintf('Missing columns in %s\n', filename);
        end
    catch ME
        fprintf('Error reading file %s: %s\n', filename, ME.message);
    end
end

%---------------------------------------
% Data validation
%---------------------------------------
nSamples = length(allPressure);
if nSamples == 0 || length(allX) ~= nSamples || length(allZ) ~= nSamples
    error('Incomplete or out-of-sync data. Check the files. CSV.');
end

%---------------------------------------
% Signal processing 
%---------------------------------------
time = linspace(0, nSamples/fs, nSamples)';

% Smooth of signal
smoothedPressure = movmean(allPressure, windowSize);
smoothedX = movmean(allX, windowSize);
smoothedZ = movmean(allZ, windowSize);


pressure_Pa = smoothedPressure * cf_Volt_Pa;

%---------------------------------------
% Plot
%---------------------------------------
figure('Name', 'Pressure / ACC_x ACC_z', 'Position', [100, 100, 800, 600]);
 m = 10;
subplot(3,1,1)
plot(time(1:m:end), smooth(pressure_Pa(1:m:end),50), 'k')
ylabel('Pressure (Pa)', 'FontSize', 12)
title('Pressure (Impact Sensor)', 'FontSize', 14)
grid on

subplot(3,1,2)
plot(time(1:m:end), smoothedX(1:m:end), 'b')
ylabel('Accel X (V)', 'FontSize', 12)
title('ACC_X', 'FontSize', 14)
grid on

subplot(3,1,3)
plot(time(1:m:end), smoothedZ(1:m:end), 'r')
xlabel('Time (s)', 'FontSize', 12)
ylabel('Accel Z (V)', 'FontSize', 12)
title('ACC_Z', 'FontSize', 14)
grid on

%---------------------------------------
% Save Data .mat
%---------------------------------------
save('CSS-9_ACCELERATION_RE_3.mat', 'time', 'smoothedX', 'smoothedZ');
save('CSS-9_IMPACT_SENSOR_RE_3.mat', 'time', 'pressure_Pa');