# license: Creative Commons Attribution 4.0 International (CC BY 4.0), http://creativecommons.org/licenses/by/4.0/
# Citation: Latypova E., Corbi F., Mastella G., Funiciello F., Moreno M., Bedford J. (2025): Data and scripts from Neighbouring segments control on earthquake recurrence patterns: Insights from scaled seismotectonic models. GFZ Data Services. https://doi.org/10.5880/fidgeo.2025.046
#
#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Jun  6 10:16:59 2025

@author: elya
"""

import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
# Scientific colour maps. Colour-vision-deficiency (CVD) friendly
from cmcrameri import cm # conda installation: https://anaconda.org/conda-forge/cmcrameri ; pip: https://pypi.org/project/cmcrameri/

############################################################
### IMPORTANT to define a working directory with all the data 
### in Linux something like: /GRL_data_scripts_latypova_et_al, 2025/data

# load data
cv_matrix_40_40Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_40Pa_asymmetric_short_fixed.csv').values
events_40_40Pa_asymmetric_short_fixed = pd.read_csv('events_40_40Pa_asymmetric_short_fixed.csv').values
cv_matrix_40_70Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_70Pa_asymmetric_short_fixed.csv').values
events_40_70Pa_asymmetric_short_fixed = pd.read_csv('events_40_70Pa_asymmetric_short_fixed.csv').values
cv_matrix_40_100Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_100Pa_asymmetric_short_fixed.csv').values
events_40_100Pa_asymmetric_short_fixed= pd.read_csv('events_40_100Pa_asymmetric_short_fixed.csv').values
cv_matrix_40_130Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_130Pa_asymmetric_short_fixed.csv').values
events_40_130Pa_asymmetric_short_fixed = pd.read_csv('events_40_130Pa_asymmetric_short_fixed.csv').values

# define the range of frame numbers
start_range = 1
end_range = 4500

total_length_cm = 145  # total length in cm along strike
n_meas_points = 25  # number of measurement points

# calculate the new x-values corresponding to the physical length
x_values_cm = np.linspace(0, total_length_cm, n_meas_points)
# define specific tick positions (for example: 15, 50, 75, and 145 cm)
tick_positions_cm = [50, 100, 145]  # the specific tick positions you want in cm
# Find the corresponding index positions in terms of measurement points (1 to 31)
tick_positions = [np.argmin(np.abs(x_values_cm - pos)) + 1 for pos in tick_positions_cm]  # Map to closest indices

# bins for hist plot
bins = [ 0.3 ,  1.42,  2.54,  3.66,  4.78,  5.9 ,  7.02,  8.14,  9.26,
       10.38, 11.5 , 12.62, 13.74, 14.86, 15.98, 17.1 , 18.22, 19.34,
       20.46, 21.58, 22.7 ]

fi = start_range  # first frame we want to analyze
la = end_range  # last frame we want to analyze

# define a vector that is threshold
tresh_vett = np.arange(0.1, 0.201, 0.001) 


fig = plt.figure(figsize=(20, 16))
gs = plt.GridSpec(4, 4, width_ratios=[5, 1, 1, 1])

letters = ['I', 'II', 'III', 'IV']
numbers = ['a', 'b', 'c', 'd']
axs = []
for i in range(4):
    for j in range(4):
        if j == 0:  
            ax = fig.add_subplot(gs[i, j])
        else:
            ax = fig.add_subplot(gs[i, j])
            
        axs.append(ax)
        
        # add the label with letter and number combination
        label = f"{letters[i]}{numbers[j]}"
        ax.set_title(label, loc='left', pad=5, x=-0.15, color='navy', fontsize=14)# fontweight='bold') 
            
axs[2].axhline(y=0.5, linewidth=2, color='k')
axs[2].axhline(y=1, linewidth=2, color='k')
axs[2].boxplot(cv_matrix_40_40Pa_asymmetric_short_fixed, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[2].set_xticks(tick_positions)                         
axs[2].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

# add rectangles
rectangles = [((8, 0.2), 4, 1, '#D95319'), ((20, 0.2), 4, 1, 'olivedrab')]

for (x, y), width, height, color in rectangles:
    # eft vertical line at x
    axs[2].plot([x, x], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)
    # right vertical line at x + width
    axs[2].plot([x + width, x + width], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)

axs[2].set_ylabel('CoV')
axs[2].set_title(f'CoV distribution')
axs[2].set_ylim([0, 1.1])

# set a main velocity threshold
threshold_main = 0.1
mat_data = events_40_40Pa_asymmetric_short_fixed

# extract the matrix data
matrix = mat_data

# mask values of the matrix that are less than or equal to 0.1
matrix_masked = np.where(matrix > threshold_main, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

### define boundaries of asperities
y1 = 7
y2 = 24.65
y3 = 16  # upper boundary of long asp
y4 = 9   # lower boundary of short asp
y5 = 6   # upper boundary of short asp

matrix_masked = matrix_masked[:, 1500:]

num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  # ticks every 10 seconds - 1 s is 50 frames

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 31, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]
axs[0].set_yticks(tick_positions)                         
axs[0].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

# plot the heatmap with specified ticks and labels, with x and y axes swapped
cax = axs[0].contourf(matrix_masked, cmap=cm.cork)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[0].set_xticks(x_ticks)
axs[0].set_xticklabels(x_labels)

# add boundaries of asperities
axs[0].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[0].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[0].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[0].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)


axs[0].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[0].text(500, 17, 'long asp 40 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[0].set_title('events above threshold 0.1 cm/s')

axs[0].set_ylabel('distance along strike, cm')

ruptures_long_array_40 = []
ruptures_short_array_40 = []
ruptures_both_array_40 = []
ruptures_other_40 = []

water = matrix_masked

for j in range(water.shape[1]):
    if np.any(water[15:25, j] > 0.1) and np.any(water[5:9, j] > 0.1):
        ruptures_both_array_40.append(j)
    elif np.any(water[15:25, j] > 0.1) and not np.any(water[5:9, j] > 0.1):
        ruptures_long_array_40.append(j)
    elif np.any(water[5:9, j] > 0.1) and not np.any(water[15:25, j] > 0.1):
        ruptures_short_array_40.append(j)
    else:
        # Check if there's any value > 0.1 outside of rows 5-9 and 15-25
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((np.arange(5, 9), np.arange(15, 25))))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_other_40.append(j)
    
    
vector_names = ['ruptures_long_array_40', 'ruptures_short_array_40', 'ruptures_both_array_40', 'ruptures_other_40']

# Initialize a list to hold the result vectors
result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)

    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)

    
ruptures_long_array_40 = result_vectors[0]
ruptures_short_array_40 = result_vectors[1]
ruptures_both_array_40 = result_vectors[2]
ruptures_other_40 = result_vectors[3]

all_ruptures = {
    'ruptures_both_array_40': ruptures_both_array_40,
    'ruptures_long_array_40': ruptures_long_array_40,
    'ruptures_short_array_40': ruptures_short_array_40,
    'ruptures_other_40': ruptures_other_40
}

def filter_close_values(primary_list, other_lists, tolerance=10):
    primary_set = set(primary_list)
    for other_list in other_lists:
        to_remove = []
        for val in other_list:
            if any(abs(val - primary_val) <= tolerance for primary_val in primary_set):
                to_remove.append(val)
    #remove close values from the other lists
        for val in to_remove:
            other_list.remove(val)
    return sorted(primary_set)

print(all_ruptures)

ruptures_both_array_40 = filter_close_values(
    all_ruptures['ruptures_both_array_40'],
    [all_ruptures['ruptures_long_array_40'], all_ruptures['ruptures_short_array_40'], all_ruptures['ruptures_other_40']]
)


ruptures_long_array_40 = filter_close_values(all_ruptures['ruptures_long_array_40'], 
                                                   [all_ruptures['ruptures_other_40']])

ruptures_short_array_40 = filter_close_values(all_ruptures['ruptures_short_array_40'], 
                                                   [all_ruptures['ruptures_other_40']])

#########################
values_list = list(all_ruptures.values())

len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

# colors for each category
colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']

# Plotting the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plotting each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[1].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # Adding labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[1].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  # Offset for the text
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

# adding labels and title and remove x-axis labels
axs[1].set_xticks([])
axs[1].set_xticklabels([])

axs[1].set_ylabel('percentage (%)')
axs[1].set_xlabel('extra load 40 Pa long asp')
axs[1].set_title('rupture classification')
axs[1].legend(loc='upper right')

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_40 = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_40 = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_40 = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[3].hist(time_diff_long_40, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[3].hist(time_diff_short_40, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[3].hist(time_diff_both_40, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[3].set_ylim(None, 25)
axs[3].set_xlim(None, 15)
# Adding labels and title
#axs[0, 3].set_xlabel('time, frames')
axs[3].set_ylabel('frequency')
axs[3].set_title('Rt distribution')
axs[3].legend()


############## 40-70 Pa experiment
start_range = 1
end_range = 4500

fi = start_range  # first frame we want to analyze
la = end_range  # last frame we want to analyze

axs[6].axhline(y=0.5, linewidth=2, color='k')
axs[6].axhline(y=1, linewidth=2, color='k')
axs[6].boxplot(cv_matrix_40_70Pa_asymmetric_short_fixed, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[6].set_xticks(tick_positions)                         
axs[6].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])           


for (x, y), width, height, color in rectangles:
    # left vertical line at x
    axs[6].plot([x, x], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)
    # right vertical line at x + width
    axs[6].plot([x + width, x + width], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)

axs[6].set_ylabel('CoV')
axs[6].set_ylim([0, 1.1])
mat_data = events_40_70Pa_asymmetric_short_fixed 

# extract the matrix data
matrix = mat_data

# mask values of the matrix that are less than or equal to threshold
matrix_masked = np.where(matrix > threshold_main, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 31, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2) 
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]



# plot the heatmap with specified ticks and labels, with x and y axes swapped
cax = axs[4].contourf(matrix_masked, cmap=cm.cork)

x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[4].set_xticks(x_ticks)
axs[4].set_xticklabels(x_labels)
axs[4].set_yticks(tick_positions)                        
axs[4].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

axs[4].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)

axs[4].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[4].text(500, 17, 'long asp 70 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[4].set_ylabel('distance along strike, cm')

ruptures_long_array_70 = []
ruptures_short_array_70 = []
ruptures_both_array_70 = []
ruptures_other_70 = []

water = matrix_masked

for j in range(water.shape[1]):
    if np.any(water[15:25, j] > 0.1) and np.any(water[5:9, j] > 0.1):
        ruptures_both_array_70.append(j)
    elif np.any(water[15:25, j] > 0.1) and not np.any(water[5:9, j] > 0.1):
        ruptures_long_array_70.append(j)
    elif np.any(water[5:9, j] > 0.1) and not np.any(water[15:25, j] > 0.1):
        ruptures_short_array_70.append(j)
    else:
        # Check if there's any value > 0.1 outside of rows 5-9 and 15-25
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((np.arange(5, 9), np.arange(15, 25))))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_other_70.append(j)
    
    

vector_names = ['ruptures_long_array_70', 'ruptures_short_array_70', 'ruptures_both_array_70', 'ruptures_other_70']

# make a list to hold the result vectors
result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)

    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    #iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # Check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)


ruptures_long_array_70 = result_vectors[0]
ruptures_short_array_70 = result_vectors[1]
ruptures_both_array_70 = result_vectors[2]
ruptures_other_70 = result_vectors[3]


all_ruptures = {
    'ruptures_both_array_70': ruptures_both_array_70,
    'ruptures_long_array_70': ruptures_long_array_70,
    'ruptures_short_array_70': ruptures_short_array_70,
    'ruptures_other_70': ruptures_other_70
}

ruptures_both_array_70 = filter_close_values(
    all_ruptures['ruptures_both_array_70'],
    [all_ruptures['ruptures_long_array_70'], all_ruptures['ruptures_short_array_70'], all_ruptures['ruptures_other_70']]
)

ruptures_long_array_70 = filter_close_values(all_ruptures['ruptures_long_array_70'], 
                                                   [all_ruptures['ruptures_other_70']])

ruptures_short_array_70 = filter_close_values(all_ruptures['ruptures_short_array_70'], 
                                                   [all_ruptures['ruptures_other_70']])

#########################
values_list = list(all_ruptures.values())

len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]
colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']

bar_width = 2  
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

#plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[5].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[5].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  # Offset for the text
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

# add labels and title and remove x-axis labels
axs[5].set_xticks([])
axs[5].set_xticklabels([])

axs[5].set_ylabel('percentage (%)')
axs[5].set_xlabel('extra load 70 Pa long asp')
plt.show()

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_70 = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_70 = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_70 = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]


axs[7].hist(time_diff_long_70, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[7].hist(time_diff_short_70, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[7].hist(time_diff_both_70, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[7].set_ylim(None, 25)
axs[7].set_xlim(None, 15)
axs[7].set_ylabel('frequency')

############# 40-100 Pa
# define the range of numbers
start_range = 1
end_range = 4500

fi = start_range  # first frame we want to analyze
la = end_range  # last frame we want to analyze

axs[10].axhline(y=0.5, linewidth=2, color='k')
axs[10].axhline(y=1, linewidth=2, color='k')
axs[10].boxplot(cv_matrix_40_100Pa_asymmetric_short_fixed, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[10].set_xticks(tick_positions)                        
axs[10].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])
for (x, y), width, height, color in rectangles:
    axs[10].plot([x, x], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)
    axs[10].plot([x + width, x + width], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)


axs[10].set_ylabel('CoV')
axs[10].set_ylim([0, 1.1])

mat_data = events_40_100Pa_asymmetric_short_fixed
matrix = mat_data

matrix_masked = np.where(matrix > threshold_main, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  # Ticks every 5 seconds

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 31, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]

# plot the heatmap with specified ticks and labels, with x and y axes swapped
cax = axs[8].contourf(matrix_masked, cmap=cm.cork)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[8].set_xticks(x_ticks)
axs[8].set_xticklabels(x_labels)
axs[8].set_yticks(tick_positions)                         # Set tick positions
axs[8].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 


axs[8].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)



# Transposing text positions
axs[8].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[8].text(500, 17, 'long asp 100 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[8].set_ylabel('distance along strike, cm')

ruptures_long_array_100 = []
ruptures_short_array_100 = []
ruptures_both_array_100 = []
ruptures_other_100 = []

water = matrix_masked

for j in range(water.shape[1]):
    if np.any(water[15:25, j] > 0.1) and np.any(water[5:9, j] > 0.1):
        ruptures_both_array_100.append(j)
    elif np.any(water[15:25, j] > 0.1) and not np.any(water[5:9, j] > 0.1):
        ruptures_long_array_100.append(j)
    elif np.any(water[5:9, j] > 0.1) and not np.any(water[15:25, j] > 0.1):
        ruptures_short_array_100.append(j)
    else:
        # Check if there's any value > 0.1 outside of rows 5-9 and 15-25
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((np.arange(5, 9), np.arange(15, 25))))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_other_100.append(j)
    
    

vector_names = ['ruptures_long_array_100', 'ruptures_short_array_100', 'ruptures_both_array_100', 'ruptures_other_100']

result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]
    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)


ruptures_long_array_100 = result_vectors[0]
ruptures_short_array_100 = result_vectors[1]
ruptures_both_array_100 = result_vectors[2]
ruptures_other_100 = result_vectors[3]
##############################################

all_ruptures = {
    'ruptures_both_array_100': ruptures_both_array_100,
    'ruptures_long_array_100': ruptures_long_array_100,
    'ruptures_short_array_100': ruptures_short_array_100,
    'ruptures_other_100': ruptures_other_100
}

ruptures_both_array_100 = filter_close_values(
    all_ruptures['ruptures_both_array_100'],
    [all_ruptures['ruptures_long_array_100'], all_ruptures['ruptures_short_array_100'], all_ruptures['ruptures_other_100']]
)

ruptures_long_array_100 = filter_close_values(all_ruptures['ruptures_long_array_100'], 
                                                   [all_ruptures['ruptures_other_100']])

ruptures_short_array_100 = filter_close_values(all_ruptures['ruptures_short_array_100'], 
                                                   [all_ruptures['ruptures_other_100']])

#########################
values_list = list(all_ruptures.values())

len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[9].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[9].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  # Offset for the text
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

# add labels and title and remove x-axis labels
axs[9].set_xticks([])
axs[9].set_xticklabels([])
axs[9].set_ylabel('percentage (%)')
axs[9].set_xlabel('extra load 100 Pa long asp')

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_100 = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_100 = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_100 = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[11].hist(time_diff_long_100, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[11].hist(time_diff_short_100, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[11].hist(time_diff_both_100, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[11].set_ylim(None, 25)
axs[11].set_xlim(None, 15)
axs[11].set_ylabel('frequency')


###################### 40-130Pa experiment

# define the range of numbers
start_range = 1
end_range = 4500

fi = start_range  # first frame we want to analyze
la = end_range  # last frame we want to analyze


axs[14].axhline(y=0.5, linewidth=2, color='k')
axs[14].axhline(y=1, linewidth=2, color='k')
axs[14].boxplot(cv_matrix_40_130Pa_asymmetric_short_fixed, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[14].set_xticks(tick_positions)                         # Set tick positions
axs[14].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])               # Set the corresponding labels


for (x, y), width, height, color in rectangles:
    axs[14].plot([x, x], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)
    axs[14].plot([x + width, x + width], [0, 1.7], color=color, linestyle='--', linewidth=1.5, zorder=1)


axs[14].set_ylabel('CoV')
axs[14].set_xlabel('distance along strike, cm')
axs[14].set_ylim([0, 1.1])

mat_data = events_40_130Pa_asymmetric_short_fixed 
# extract the matrix data
matrix = mat_data
# mask values of the matrix that are less than or equal to 0.1
matrix_masked = np.where(matrix > threshold_main, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  # Ticks every 5 seconds

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 31, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]

# plot the heatmap with specified ticks and labels, with x and y axes swapped
cax = axs[12].contourf(matrix_masked, cmap=cm.cork)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[12].set_xticks(x_ticks)
axs[12].set_xticklabels(x_labels)
axs[12].set_yticks(tick_positions)
axs[12].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

axs[12].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)

axs[12].axvline(x=25, color='goldenrod', linestyle='--', linewidth=1.5)
axs[12].axvline(x=550, color='goldenrod', linestyle='--', linewidth=1.5)


axs[12].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[12].text(500, 17, 'long asp 130 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

#axs[3, 0].set_title('events above threshold 0.1 cm/s')
axs[12].set_ylabel('distance along strike, cm')
axs[12].set_xlabel('time, seconds')

ruptures_long_array_130 = []
ruptures_short_array_130 = []
ruptures_both_array_130 = []
ruptures_other_130 = []

water = matrix_masked

for j in range(water.shape[1]):
    if np.any(water[15:25, j] > 0.1) and np.any(water[5:9, j] > 0.1):
        ruptures_both_array_130.append(j)
    elif np.any(water[15:25, j] > 0.1) and not np.any(water[5:9, j] > 0.1):
        ruptures_long_array_130.append(j)
    elif np.any(water[5:9, j] > 0.1) and not np.any(water[15:25, j] > 0.1):
        ruptures_short_array_130.append(j)
    else:
        # Check if there's any value > 0.1 outside of rows 5-9 and 15-25
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((np.arange(5, 9), np.arange(15, 25))))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_other_130.append(j)
    

vector_names = ['ruptures_long_array_130', 'ruptures_short_array_130', 'ruptures_both_array_130', 'ruptures_other_130']

result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)
    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)

ruptures_long_array_130 = result_vectors[0]
ruptures_short_array_130 = result_vectors[1]
ruptures_both_array_130 = result_vectors[2]
ruptures_other_130 = result_vectors[3]
##############################################

all_ruptures = {
    'ruptures_both_array_130': ruptures_both_array_130,
    'ruptures_long_array_130': ruptures_long_array_130,
    'ruptures_short_array_130': ruptures_short_array_130,
    'ruptures_other_130': ruptures_other_130
}

ruptures_both_array_130 = filter_close_values(
    all_ruptures['ruptures_both_array_130'],
    [all_ruptures['ruptures_long_array_130'], all_ruptures['ruptures_short_array_130'], all_ruptures['ruptures_other_130']]
)

ruptures_long_array_130 = filter_close_values(all_ruptures['ruptures_long_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])

ruptures_short_array_130 = filter_close_values(all_ruptures['ruptures_short_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])

#########################
values_list = list(all_ruptures.values())

len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

# plot the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[13].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[13].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  # Offset for the text
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

# add labels and title and remove x-axis labels
axs[13].set_xticks([])
axs[13].set_xticklabels([])

axs[13].set_ylabel('percentage (%)')
axs[13].set_xlabel('extra load 130 Pa long asp')

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_130 = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_130 = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_130 = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[15].hist(time_diff_long_130, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[15].hist(time_diff_short_130, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[15].hist(time_diff_both_130, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[15].set_ylim(None, 25)
axs[15].set_xlim(None, 15)

axs[15].set_xlabel('time, seconds')
axs[15].set_ylabel('frequency')


fig.tight_layout(pad=3.0, h_pad=2.0, w_pad=2.0)

axs[3].sharex(axs[15])
axs[7].sharex(axs[15])

axs[11].sharex(axs[15])

plt.show()