# license: Creative Commons Attribution 4.0 International (CC BY 4.0), http://creativecommons.org/licenses/by/4.0/
# Citation: Latypova E., Corbi F., Mastella G., Funiciello F., Moreno M., Bedford J. (2025): Data and scripts from Neighbouring segments control on earthquake recurrence patterns: Insights from scaled seismotectonic models. GFZ Data Services. https://doi.org/10.5880/fidgeo.2025.046
#
#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Jun  6 14:11:57 2025

@author: elya
"""

import matplotlib.pyplot as plt
import numpy as np
# Scientific colour maps. Colour-vision-deficiency (CVD) friendly
from cmcrameri import cm # conda installation: https://anaconda.org/conda-forge/cmcrameri ; pip: https://pypi.org/project/cmcrameri/
import pandas as pd
############################################################
### IMPORTANT to define a working directory with all the data 
### in Linux something like: /GRL_data_scripts_latypova_et_al, 2025/data

# load data
cv_matrix_40Pa_single = pd.read_csv('cv_matrix_40Pa_single.csv').values
events_40Pa_single = pd.read_csv('events_40Pa_single.csv').values
cv_matrix_70Pa_single  = pd.read_csv('cv_matrix_70Pa_single.csv').values
events_70Pa_single = pd.read_csv('events_70Pa_single.csv').values
cv_matrix_100Pa_single = pd.read_csv('cv_matrix_100Pa_single.csv').values
events_100Pa_single = pd.read_csv('events_100Pa_single.csv').values
cv_matrix_130Pa_single = pd.read_csv('cv_matrix_130Pa_single.csv').values
events_130Pa_single = pd.read_csv('events_130Pa_single.csv').values

# define the range of numbers
start_range = 1
end_range = 4500

total_length_cm = 145  # total length of x coordinate in cm 
n_meas_points = 25  # number of measurement points in data matrix
# calculate the new x-values corresponding to the physical length
x_values_cm = np.linspace(0, total_length_cm, n_meas_points)
# define specific tick positions (for example: 15, 50, 75, and 145 cm)
tick_positions_cm = [50, 100, 145]  # the specific tick positions you want in cm
# find the corresponding index positions in terms of measurement points (1 to 31)
tick_positions = [np.argmin(np.abs(x_values_cm - pos)) + 1 for pos in tick_positions_cm]  # Map to closest indices

fig = plt.figure(figsize=(20, 15))# dpi=120)
gs = plt.GridSpec(4, 4, width_ratios=[5, 1, 1, 1])

letters = ['I', 'II', 'III', 'IV']
numbers = ['a', 'b', 'c', 'd']
axs = []
for i in range(4):
    for j in range(4):
        if j == 0:  # For the first column (1, 5, 9, 13)
            ax = fig.add_subplot(gs[i, j])
        else:
            ax = fig.add_subplot(gs[i, j])
            
        axs.append(ax)

        label = f"{letters[i]}{numbers[j]}"
        ax.set_title(label, loc='left', pad=5, x=-0.12, color='navy', fontsize=14)# fontweight='bold') 
    

axs[2].axhline(y=0.5, linewidth=2, color='k')
axs[2].axhline(y=1, linewidth=2, color='k')
axs[2].boxplot(cv_matrix_40Pa_single, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[2].set_xticks(tick_positions)                         # Set tick positions
axs[2].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])    
# add rectangles to the subplot
rectangles = [
    ((11, 0.2), 4, 1, 'olivedrab')
    ] # short asp

for (x, y), width, height, color in rectangles:
    # left vertical line at x
    axs[2].plot([x, x], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)
    # right vertical line at x + width
    axs[2].plot([x + width, x + width], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)

axs[2].set_ylabel('CoV')
axs[2].set_title(f'CoV distribution')
axs[2].set_ylim([-0.1, 1.4])
### define event data file
mat_data = events_40Pa_single
matrix = mat_data
matrix_masked = np.where(matrix > 0.1, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

y4 = 15   # lower boundary of single asp
y5 = 11   # upper boundary of single asp
matrix_masked = matrix_masked[:, 1500:]

num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  # Ticks every 5 seconds

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 25, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]
axs[0].set_yticks(tick_positions)                         # Set tick positions
axs[0].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

cax = axs[0].contourf(matrix_masked, cmap=cm.cork)

x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[0].set_xticks(x_ticks)
axs[0].set_xticklabels(x_labels)

axs[0].axhline(y=y4, color='olivedrab', linestyle='--', linewidth=1.5)
axs[0].axhline(y=y5, color='olivedrab', linestyle='--', linewidth=1.5)

axs[0].text(500, 12, 'single asp 40 Pa', color='olivedrab', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[0].set_title('events above threshold 0.1 cm/s')

axs[0].set_ylabel('distance along strike')

ruptures_single_40 = []
ruptures_single_other_40 = []

water = matrix_masked
for j in range(water.shape[1]):
    if np.any(water[11:15, j] > 0.1):
        ruptures_single_40.append(j)    
    else:
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.arange(11, 15))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_single_other_40.append(j)


vector_names = ['ruptures_single_40', 'ruptures_single_other_40']
# create a list to hold the result vectors
result_vectors = []
for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)

    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)

    # display the result vector
    print(f"result for {vector_name}:")
    print(result_vector)
    
ruptures_single_40 = result_vectors[0]
ruptures_other_40 = result_vectors[1]

all_ruptures = {
    'ruptures_single_40': ruptures_single_40,
    'ruptures_other_40': ruptures_other_40
}

def filter_close_values(primary_list, other_lists, tolerance=10):
    primary_set = set(primary_list)
    for other_list in other_lists:
        to_remove = []
        for val in other_list:
            if any(abs(val - primary_val) <= tolerance for primary_val in primary_set):
                to_remove.append(val)
    #remove close values from the other lists
        for val in to_remove:
            other_list.remove(val)
    return sorted(primary_set)


ruptures_single = filter_close_values(
    all_ruptures['ruptures_single_40'],
    [
     all_ruptures['ruptures_other_40']]
)
#########################
values_list = list(all_ruptures.values())

len_single = len(result_vectors[0])
len_other = len(result_vectors[1])

categories = ['single', 'other']
values = [len_single, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

colors = ['olivedrab','plum']

# plot the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[1].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[1].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

# add labels and title and remove x-axis labels
axs[1].set_xticks([])
axs[1].set_xticklabels([])

axs[1].set_ylabel('percentage (%)')
axs[1].set_xlabel('extra load 40 Pa')
axs[1].set_title('rupture classification')
axs[1].legend(loc='upper right')

plt.show()
#####

################### recurrence time for central points of asperities
time_single = values_list[0]
time_single  = [t / 50 for t in time_single ]
time_diff_single_40 = [time_single[i] - time_single[i-1] for i in range(1, len(time_single))]

# bins for hist plot
bins = [0.58      , 0.71933333, 0.85866667, 0.998     , 1.13733333,
       1.27666667, 1.416     , 1.55533333, 1.69466667, 1.834     ,
       1.97333333, 2.11266667, 2.252     , 2.39133333, 2.53066667,
       2.67      , 2.80933333, 2.94866667, 3.088     , 3.22733333,
       3.36666667, 3.506     , 3.64533333, 3.78466667, 3.924     ,
       4.06333333, 4.20266667, 4.342     , 4.48133333, 4.62066667,
       4.76      ]
axs[3].hist(time_diff_single_40, bins=bins, alpha=0.5, label='single asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[3].set_ylim(None, 17)
axs[3].set_xlim(None, 5)

axs[3].set_ylabel('frequency')
axs[3].set_title('Rt distribution')
axs[3].legend()

############## 70 Pa

axs[6].axhline(y=0.5, linewidth=2, color='k')
axs[6].axhline(y=1, linewidth=2, color='k')
axs[6].boxplot(cv_matrix_70Pa_single , labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})
axs[6].set_xticks(tick_positions)                         # Set tick positions
axs[6].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])    
    
for (x, y), width, height, color in rectangles:
    axs[6].plot([x, x], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)
    axs[6].plot([x + width, x + width], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)


axs[6].set_ylabel('CoV')
axs[6].set_ylim([-0.1, 1.4])

mat_data = events_70Pa_single 
# extract the matrix data
matrix = mat_data

# mask values of the matrix that are less than or equal to 0.1
matrix_masked = np.where(matrix > 0.1, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  # Ticks every 5 seconds

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 25, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]
axs[4].set_yticks(tick_positions)                        
axs[4].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 

cax = axs[4].contourf(matrix_masked, cmap=cm.cork)

x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[4].set_xticks(x_ticks)
axs[4].set_xticklabels(x_labels)

axs[4].axhline(y=y4, color='olivedrab', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y5, color='olivedrab', linestyle='--', linewidth=1.5)

axs[4].text(500, 12, 'single asp 70 Pa', color='olivedrab', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[4].set_ylabel('distance along strike')

ruptures_single_70 = []
ruptures_single_other_70 = []

water = matrix_masked
for j in range(water.shape[1]):
    if np.any(water[11:15, j] > 0.1):
        ruptures_single_70.append(j)    
    else:
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.arange(11, 15))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_single_other_70.append(j)

vector_names = ['ruptures_single_70', 'ruptures_single_other_70']

result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)

    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)
    # display the result vector
    print(f"result for {vector_name}:")
    print(result_vector)
    
ruptures_single_70 = result_vectors[0]
ruptures_other_70 = result_vectors[1]

all_ruptures = {
    'ruptures_single_70': ruptures_single_70,
    'ruptures_other_70': ruptures_other_70
}

ruptures_single = filter_close_values(
    all_ruptures['ruptures_single_70'],
    [
     all_ruptures['ruptures_other_70']]
)

#########################
values_list = list(all_ruptures.values())
len_single = len(result_vectors[0])
len_other = len(result_vectors[1])

categories = ['single', 'other']
values = [len_single, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]


# plot the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# Plotting each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[5].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[5].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')


axs[5].set_xticks([])
axs[5].set_xticklabels([])
axs[5].set_xlabel('extra load 70 Pa')
axs[5].set_ylabel('percentage (%)')

################### recurrence time for central points of asperities
time_single = values_list[0]
time_single  = [t / 50 for t in time_single ]
time_diff_single_70 = [time_single[i] - time_single[i-1] for i in range(1, len(time_single))]

axs[7].hist(time_diff_single_70, bins=bins, alpha=0.5, label='single asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[7].set_ylim(None, 17)
axs[7].set_xlim(None, 5)
axs[7].set_ylabel('frequency')

########################### 100 Pa

axs[10].axhline(y=0.5, linewidth=2, color='k')
axs[10].axhline(y=1, linewidth=2, color='k')
axs[10].boxplot(cv_matrix_100Pa_single, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[10].set_xticks(tick_positions)                         
axs[10].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])    

for (x, y), width, height, color in rectangles:
    # left vertical line at x
    axs[10].plot([x, x], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)
    # right vertical line at x + width
    axs[10].plot([x + width, x + width], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)


axs[10].set_ylabel('CoV')
axs[10].set_ylim([-0.1, 1.4])

mat_data = events_100Pa_single
matrix = mat_data
# mask values of the matrix that are less than or equal to 0.1
matrix_masked = np.where(matrix > 0.1, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 25, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]
axs[8].set_yticks(tick_positions)                         
axs[8].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 


# plot the heatmap with specified ticks and labels, with x and y axes swapped
cax = axs[8].contourf(matrix_masked, cmap=cm.cork)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[8].set_xticks(x_ticks)
axs[8].set_xticklabels(x_labels)
axs[8].axhline(y=y4, color='olivedrab', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y5, color='olivedrab', linestyle='--', linewidth=1.5)

axs[8].text(500, 12, 'single asp 100 Pa', color='olivedrab', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[8].set_ylabel('distance along strike')

ruptures_single_100 = []
ruptures_single_other_100 = []

water = matrix_masked
for j in range(water.shape[1]):
    if np.any(water[11:15, j] > 0.1):
        ruptures_single_100.append(j)    
    else:
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.arange(11, 15))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_single_other_100.append(j)

vector_names = ['ruptures_single_100', 'ruptures_single_other_100']

# create a list to hold the result vectors
result_vectors = []
for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)
    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)

    print(f"result for {vector_name}:")
    print(result_vector)

ruptures_single_100 = result_vectors[0]
ruptures_other_100 = result_vectors[1]

all_ruptures = {
    'ruptures_single_100': ruptures_single_100,
    'ruptures_other_100': ruptures_other_100
}


ruptures_single = filter_close_values(
    all_ruptures['ruptures_single_100'],
    [
     all_ruptures['ruptures_other_100']]
)

#########################
values_list = list(all_ruptures.values())
len_single = len(result_vectors[0])
len_other = len(result_vectors[1])

categories = ['single', 'other']
values = [len_single, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

# plot the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[9].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[9].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')


axs[9].set_xticks([])
axs[9].set_xticklabels([])

axs[9].set_xlabel('extra load 100 Pa')
axs[9].set_ylabel('percentage (%)')

################### Recurrence time for central points of asperities
time_single = values_list[0]
time_single  = [t / 50 for t in time_single ]
time_diff_single_100 = [time_single[i] - time_single[i-1] for i in range(1, len(time_single))]

axs[11].hist(time_diff_single_100, bins=bins, alpha=0.5, label='single asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[11].set_ylim(None, 17)
axs[11].set_xlim(None, 5)
axs[11].set_ylabel('frequency')

###################### 130 Pa

axs[14].axhline(y=0.5, linewidth=2, color='k')
axs[14].axhline(y=1, linewidth=2, color='k')
axs[14].boxplot(cv_matrix_130Pa_single, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})

axs[14].set_xticks(tick_positions)                         # Set tick positions
axs[14].set_xticklabels([f'{int(pos)}' for pos in tick_positions_cm])    

for (x, y), width, height, color in rectangles:
    axs[14].plot([x, x], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)
    axs[14].plot([x + width, x + width], [-0.1, 1.4], color=color, linestyle='--', linewidth=1.5, zorder=1)


axs[14].set_ylabel('CoV')
axs[14].set_xlabel('distance along strike')

axs[14].set_ylim([-0.1, 1.4])

#define events data
mat_data = events_130Pa_single
matrix = mat_data

# mask values of the matrix that are less than or equal to 0.1 - velocity  threshold
matrix_masked = np.where(matrix > 0.1, matrix, np.nan)
matrix_zeros = matrix_masked[np.isnan(matrix_masked)] = 0

matrix_masked = matrix_masked[:, 1500:]

# settings for ticks on the x-axis
num_columns = matrix_masked.shape[1]
x_seconds = np.linspace(0, num_columns, num_columns)
x_ticks = np.arange(0, num_columns, 500)  

# settings for ticks on the y-axis
num_rows = matrix_masked.shape[0]
y_cm = np.linspace(0, 25, num_rows)
y_ticks = np.linspace(0, num_rows - 1, 2)  # Ticks every 10 cm
y_labels = [f"{y:.0f}" for y in np.linspace(0, 145, 15)]
axs[12].set_yticks(tick_positions)                         # Set tick positions
axs[12].set_yticklabels([f'{int(pos)}' for pos in tick_positions_cm]) 
cax = axs[12].contourf(matrix_masked, cmap=cm.cork)

x_labels = [f"{tick/50:.1f}" for tick in x_ticks] 
axs[12].set_xticks(x_ticks)
axs[12].set_xticklabels(x_labels)

axs[12].axhline(y=y4, color='olivedrab', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y5, color='olivedrab', linestyle='--', linewidth=1.5)

axs[12].text(500, 12, 'single asp 130 Pa', color='olivedrab', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[12].set_ylabel('matrix rows (distance along strike)')
axs[12].set_xlabel('time, seconds')

ruptures_single_130 = []
ruptures_single_other_130 = []

water = matrix_masked
for j in range(water.shape[1]):
    if np.any(water[11:15, j] > 0.1):
        ruptures_single_130.append(j)    
    else:
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.arange(11, 15))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_single_other_130.append(j)


vector_names = ['ruptures_single_130', 'ruptures_single_other_130']

result_vectors = []

for vector_name in vector_names:
    # get the current vector using eval
    current_vector = eval(vector_name)

    # initialize the result vector with the first element of the current vector
    if len(current_vector) == 0:
        result_vector = []
    else:
        result_vector = [current_vector[0]]

    # iterate through the current vector starting from the second element
    for i in range(1, len(current_vector)):
        # check if the difference between the current element and the last element
        # in the result vector is greater than 10
        if (current_vector[i] - result_vector[-1]) > 10:
            # append the current element to the result vector
            result_vector.append(current_vector[i])

    # store the result vector in the list
    result_vectors.append(result_vector)

    # display the result vector
    print(f"result for {vector_name}:")
    print(result_vector)

ruptures_single_130 = result_vectors[0]
ruptures_other_130 = result_vectors[1]

all_ruptures = {
    'ruptures_single_130': ruptures_single_130,
    'ruptures_other_130': ruptures_other_130
}

ruptures_single = filter_close_values(
    all_ruptures['ruptures_single_130'],
    [
     all_ruptures['ruptures_other_130']]
)

#########################
values_list = list(all_ruptures.values())
len_single = len(result_vectors[0])
len_other = len(result_vectors[1])

categories = ['single', 'other']
values = [len_single, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]


# plot the stacked bar chart with annotations
bar_width = 2  # Consistent bar width
bars = []

# calculate x positions for each bar
x_positions = [i * (bar_width + 1) for i in range(len(percentages))]

# plot each bar separately
for idx, (pct, color, x_pos) in enumerate(zip(percentages, colors, x_positions)):
    bar = axs[13].bar(x_pos, pct, color=color, label=categories[idx], width=bar_width)

    # add labels directly on the bars
    for rect in bar:
        height = rect.get_height()
        axs[13].annotate(f'{height:.1f}%',
                    xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 2),
                    xytext=(0, 0),  
                    textcoords="offset points",
                    ha='center', va='center',
                    fontsize=9, color='black', weight='bold')

axs[13].set_xticks([])
axs[13].set_xticklabels([])
axs[13].set_xlabel('extra load 130 Pa')
axs[13].set_ylabel('percentage (%)')


################### recurrence time for central points of asperities
time_single = values_list[0]
time_single  = [t / 50 for t in time_single ]
time_diff_single_130 = [time_single[i] - time_single[i-1] for i in range(1, len(time_single))]
axs[15].hist(time_diff_single_130, bins=bins, alpha=0.5, label='single asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[15].set_ylim(None, 17)
axs[15].set_xlim(None, 5)
axs[15].set_xlabel('time, seconds')
axs[15].set_ylabel('frequency')

fig.tight_layout(pad=3.0, h_pad=2.0, w_pad=2.0)

axs[7].sharex(axs[3])
axs[11].sharex(axs[3])
axs[15].sharex(axs[3])
plt.show()