# license: Creative Commons Attribution 4.0 International (CC BY 4.0), http://creativecommons.org/licenses/by/4.0/
# Citation: Latypova E., Corbi F., Mastella G., Funiciello F., Moreno M., Bedford J. (2025): Data and scripts from Neighbouring segments control on earthquake recurrence patterns: Insights from scaled seismotectonic models. GFZ Data Services. https://doi.org/10.5880/fidgeo.2025.046
#
#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Mar 13 15:34:03 2025

@author: elya
"""
import matplotlib.pyplot as plt     # https://matplotlib.org/stable/install/index.html
import numpy as np  #https://numpy.org/install/
import pandas as pd

# Scientific colour maps. Colour-vision-deficiency (CVD) friendly
from cmcrameri import cm # conda installation: https://anaconda.org/conda-forge/cmcrameri ; pip: https://pypi.org/project/cmcrameri/
from matplotlib.patches import FancyArrowPatch
############################################################
### IMPORTANT to define a working directory with all the data 
### in Linux something like: /GRL_data_scripts_latypova_et_al, 2025/data

# function definitions
#### function to plot CoV boxplots along the strike
def plot_boxplot(ax, data, n_meas_points, tick_positions, tick_labels, rectangles, ylabel, ylim):
    """plot a boxplot for rupture classification"""
    ax.axhline(y=0.5, linewidth=1, color='k')
    ax.axhline(y=1, linewidth=1, color='k')
    ax.boxplot(data, labels=[f'{p + 1}' for p in range(n_meas_points)], flierprops={'marker': 'o', 'markersize': 4, 'markerfacecolor': 'fuchsia'})
    ax.set_xticks(tick_positions)
    ax.set_xticklabels(tick_labels)
    for (x, y), width, height, color in rectangles:
        #left vertical line at x
        ax.plot([x, x], [-0.2, 1.1], color=color, linestyle='--', linewidth=1.5, zorder=1)
        # right vertical line at x + width
        ax.plot([x + width, x + width], [-0.2, 1.1], color=color, linestyle='--', linewidth=1.5, zorder=1)
    
    ax.set_ylabel(ylabel)
    ax.set_ylim(ylim)

#### function to plot timeline maps
def plot_heatmap(ax, data, tick_positions, tick_labels, x_ticks, x_labels, title, y_label):
    """plot a heatmap of events above a threshold"""
    cax = ax.contourf(data, cmap=cm.cork)
    ax.set_xticks(x_ticks)
    ax.set_xticklabels(x_labels)
    ax.set_yticks(tick_positions)
    ax.set_yticklabels(tick_labels)
    ax.set_title(title)
    ax.set_ylabel(y_label)

#### function to process events vectors above a threshold and to define an events 
#with sensitivity of 0.2 seconds which is 10 time frames (classification o ruptures - where is happening) 
#for each segment 
def process_vectors(vector_names):
   
    result_vectors = []
    for vector_name in vector_names:
        # get the current vector using eval
        current_vector = eval(vector_name)

    
        if len(current_vector) == 0:
            result_vector = []
        else:
            result_vector = [current_vector[0]]

        # iterate through the current vector 
        for i in range(1, len(current_vector)):
            # check if the difference between the current element and the last element
            # in the result vector is greater than 10 frames - 0.2 seconds
            if (current_vector[i] - result_vector[-1]) > 10:
                
                result_vector.append(current_vector[i])

        result_vectors.append(result_vector)

    
    return result_vectors

######## function to make a classification of ruptures 
# after the process_vectors applied - to be sure there are no overlapping events in different segments
# for example both asperities has events [109, 190, 250,...]
#asp1 has events [110, 157, 220,...], asp2 has events [112, 235, 270,...]
# so events in the both asperities list checking other lists to find events within 10 time frames
# if so the function remove close events because there is 1 event according to our classification 
#with the sensitivity of 10 frames or 0.2 seconds.

def filter_close_values(primary_list, other_lists, tolerance=10):
    primary_set = set(primary_list)
    for other_list in other_lists:
        to_remove = []
        for val in other_list:
            if any(abs(val - primary_val) <= tolerance for primary_val in primary_set):
                to_remove.append(val)
    #remove close values from the other lists
        for val in to_remove:
            other_list.remove(val)
    return sorted(primary_set)

#### function to plot Rt distribution 
def plot_bar_chart(ax, categories, values, colors, bar_width=2):
    """plot a bar chart with annotations for Rt"""
    x_positions = [i * (bar_width + 1) for i in range(len(values))]
    for idx, (value, color, x_pos) in enumerate(zip(values, colors, x_positions)):
        bar = ax.bar(x_pos, value, color=color, label=categories[idx], width=bar_width)
        for rect in bar:
            height = rect.get_height()
            ax.annotate(f'{height:.1f}%'
                        , xy=(rect.get_x() + rect.get_width() / 2, rect.get_y() + height / 1.8),
                        xytext=(0, 0),
                        textcoords="offset points",
                        ha='center', va='center',
                        fontsize=8, color='black', weight='bold')
    ax.set_xticks([])
    ax.set_xticklabels([])


# load data
cv_matrix_130Pa_single = pd.read_csv('cv_matrix_130Pa_single.csv').values 
events_130Pa_single = pd.read_csv('events_130Pa_single.csv').values 
cv_matrix_40_100Pa_symmetric_exp = pd.read_csv('cv_matrix_40_100Pa_symmetric.csv').values 
events_40_100Pa_symmetric = pd.read_csv('events_40_100Pa_symmetric.csv').values 
cv_matrix_40_40Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_40Pa_asymmetric_short_fixed.csv').values 
events_40_40Pa_asymmetric_short_fixed = pd.read_csv('events_40_40Pa_asymmetric_short_fixed.csv').values 
cv_matrix_40_130Pa_asymmetric_short_fixed = pd.read_csv('cv_matrix_40_130Pa_asymmetric_short_fixed.csv').values 
events_40_130Pa_asymmetric_short_fixed = pd.read_csv('events_40_130Pa_asymmetric_short_fixed.csv').values 
cv_matrix_40_70Pa_asymmetric_long_fixed = pd.read_csv('cv_matrix_40_70Pa_asymmetric_long_fixed.csv').values 
events_40_70Pa_asymmetric_long_fixed = pd.read_csv('events_40_70Pa_asymmetric_long_fixed.csv').values 
cv_matrix_40_130Pa_asymmetric_long_fixed = pd.read_csv('cv_matrix_40_130Pa_asymmetric_long_fixed.csv').values 
events_40_130Pa_asymmetric_long_fixed = pd.read_csv('events_40_130Pa_asymmetric_long_fixed.csv').values 



# define variables
total_length_cm = 145 # length of the foamquake
n_meas_points = 25 # matrix columns from the original .mat files
x_values_cm = np.linspace(0, total_length_cm, n_meas_points)
tick_positions_cm = [50, 100, 145]
tick_positions = [np.argmin(np.abs(x_values_cm - pos)) + 1 for pos in tick_positions_cm]

# plot subplots
fig = plt.figure(figsize=(20, 16))
gs = plt.GridSpec(6, 4, width_ratios=[5, 1, 1, 1])
letters = ['I', 'II', 'III', 'IV', 'V', 'VI']
numbers = ['a', 'b', 'c', 'd']
axs = []
for i in range(6):
    for j in range(4):
        ax = fig.add_subplot(gs[i, j])
        axs.append(ax)
        label = f"{letters[i]}{numbers[j]}"
        ax.set_title(label, loc='left', pad=5, x=-0.15, color='navy', fontsize=14)

### lines to separate diff configurations

arrow_line = FancyArrowPatch(
    (0.03, 0.822),  # start point (x, y)
    (0.03, 0.996),  # end point (x, y)
    arrowstyle='<|-|>',  
    color='olivedrab',
    linewidth=1.5,
    mutation_scale=15
)
fig.add_artist(arrow_line)


line = plt.Line2D([0.03, 0.03], [0.822, 0.996], color='green', transform=fig.transFigure,
                  linewidth=1.5, linestyle='--')

fig.text(0.02, 0.9, 'single', ha='center', va='center', fontsize=12, rotation=90)   

arrow_line = FancyArrowPatch(
    (0.03, 0.665),  # start point (x, y)
    (0.03, 0.82),  # end point (x, y)
    arrowstyle='<|-|>',  
    color='orange',
    linewidth=1.5,
    mutation_scale=15
)
fig.add_artist(arrow_line)

fig.text(0.02, 0.74, 'symmetric', ha='center', va='center', fontsize=12, rotation=90) 

arrow_line = FancyArrowPatch(
    (0.03, 0.05),  # start point (x, y)
    (0.03, 0.66),  # end point (x, y)
    arrowstyle='<|-|>',  
    color='purple',
    linewidth=1.5,
    mutation_scale=15
)
fig.add_artist(arrow_line)
fig.text(0.02, 0.4, 'asymmetric', ha='center', va='center', fontsize=12, rotation=90) 

################# SINGLE ASP EXPERIMENT
# plot boxplot with CoV for single asp exps (axs[2])
rectangles = [((11, 0.1), 5, 1.1, 'olivedrab')]

plot_boxplot(axs[2], cv_matrix_130Pa_single, n_meas_points, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [-0.2, 1.1])
axs[2].set_title('CoV distribution')
axs[2].axhline(y=11, color='olivedrab', linestyle='--', linewidth=1.5)
axs[2].axhline(y=16, color='olivedrab', linestyle='--', linewidth=1.5)
# plot timeline map (axs[0])
x_ticks = np.arange(0, events_130Pa_single[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[0], events_130Pa_single[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')

#### coordinates from .mat files
y3 = 16  # upper boundary of long asp
y4 = 15   # lower boundary of short asp
y5 = 11   # upper boundary of short asp


axs[0].axhline(y=y4, color='#77AC30', linestyle='--', linewidth=1.5)
axs[0].axhline(y=y5, color='#77AC30', linestyle='--', linewidth=1.5)
axs[0].text(500, 12, 'single asp 130 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

# plot rupture classification (axs[1])
water = events_130Pa_single[:, 1500:]
# compute Rt - recurrence time
excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.arange(11, 15))
ruptures_single_130 = [j for j in range(water.shape[1]) if np.any(water[11:15, j] > 0.1)]
ruptures_other_130 = [j for j in range(water.shape[1]) if j not in ruptures_single_130 and np.any(water[excluded_rows, j] > 0.1)]

vector_names = {'ruptures_single_130': ruptures_single_130, 'ruptures_other_130': ruptures_other_130}
result_vectors = process_vectors(vector_names)

ruptures_single_130 = result_vectors[0]
ruptures_other_130 = result_vectors[1]

all_ruptures = {
    'ruptures_single_130': ruptures_single_130,
    'ruptures_other_130': ruptures_other_130
}

ruptures_single = filter_close_values(
    all_ruptures['ruptures_single_130'],
    [
     all_ruptures['ruptures_other_130']]
)


values_list = list(all_ruptures.values())
len_single = len(result_vectors[0])
len_other = len(result_vectors[1])

categories = ['single', 'other']
values = [len_single, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

#colors for each category
colors = ['olivedrab', 'plum']
# plot ruprure classification bar chart
plot_bar_chart(axs[1], categories, percentages, colors)
axs[1].set_xlabel('extra load 130 Pa')
axs[1].set_ylabel('percentage (%)')
axs[1].set_title('rupture classification')
axs[1].legend(loc='upper right', fontsize=9, framealpha=0.7)

# plot Rt distribution hist for central point of single asp  (axs[3])
time_single = values_list[0]
time_single  = [t / 50 for t in time_single ]
time_diff_single = [time_single[i] - time_single[i-1] for i in range(1, len(time_single))]
bins = [ 0.3 ,  1.42,  2.54,  3.66,  4.78,  5.9 ,  7.02,  8.14,  9.26,
       10.38, 11.5 , 12.62, 13.74, 14.86, 15.98, 17.1 , 18.22, 19.34,
       20.46, 21.58, 22.7 ]

axs[3].hist(time_diff_single, bins=bins, alpha=0.5, label='single asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[3].set_title('Rt distribution')
axs[3].set_ylabel('frequency')

axs[3].set_ylim(None, 31)
axs[3].set_xlim(None, 17)
axs[3].legend()

################################################# DOUBLE SYMMETRIC ASP EXPS
# plot boxplot with CoV for double symmetric asp exps
rectangles = [((8, 0.1), 4, 0.55, '#D95319'), ((20, 0.1), 4, 0.55, 'olivedrab')]
plot_boxplot(axs[6], cv_matrix_40_100Pa_symmetric_exp, 31, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [0, 1.1])
# plot timeline map (axs[4])
x_ticks = np.arange(0, events_40_100Pa_symmetric[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[4], events_40_100Pa_symmetric[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')

y2 = 24
y3 = 20  # upper boundary of long asp
y4 = 12   # lower boundary of short asp
y5 = 8   # upper boundary of short asp
axs[4].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)#77AC30
axs[4].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[4].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)
axs[4].text(500, 8, 'asp1 100 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[4].text(500, 20, 'asp2 fixed 40 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

# rupture classification for double symmetric asperities experiment
water = events_40_100Pa_symmetric[:, 1500:]
threshold_main = 0.1
# define row ranges for asperities - this is the asperity boundaries
asp1_rows = np.arange(7, 12)
asp2_rows = np.arange(19, 24)
excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((asp1_rows, asp2_rows)))
### define events within asperities and outside
ruptures_both_array_100 = [
    j for j in range(water.shape[1]) 
    if np.any(water[asp1_rows, j] > threshold_main) and np.any(water[asp2_rows, j] > threshold_main)
]

ruptures_asp1_array_100 = [
    j for j in range(water.shape[1]) 
    if np.any(water[asp1_rows, j] > threshold_main) and not np.any(water[asp2_rows, j] > threshold_main)
]

ruptures_asp2_array_100 = [
    j for j in range(water.shape[1]) 
    if np.any(water[asp2_rows, j] > threshold_main) and not np.any(water[asp1_rows, j] > threshold_main)
]

ruptures_other_100 = [
    j for j in range(water.shape[1]) 
    if not np.any(water[asp1_rows, j] > threshold_main) 
    and not np.any(water[asp2_rows, j] > threshold_main) 
    and np.any(water[excluded_rows, j] > threshold_main)
]

vector_names = {'ruptures_asp1_array_100': ruptures_asp1_array_100,'ruptures_asp2_array_100': ruptures_asp2_array_100,
                'ruptures_both_array_100': ruptures_both_array_100, 'ruptures_other_100': ruptures_other_100}

result_vectors = process_vectors(vector_names)

ruptures_asp1_array_100 = result_vectors[0]
ruptures_asp2_array_100 = result_vectors[1]
ruptures_both_array_100 = result_vectors[2]
ruptures_other_100 = result_vectors[3]

all_ruptures = {
    'ruptures_both_array_100': ruptures_both_array_100,
    'ruptures_asp1_array_100': ruptures_asp1_array_100,
    'ruptures_asp2_array_100': ruptures_asp2_array_100,
    'ruptures_other_100': ruptures_other_100
}



ruptures_both_array_100 = filter_close_values(
    all_ruptures['ruptures_both_array_100'],
    [all_ruptures['ruptures_asp1_array_100'], all_ruptures['ruptures_asp2_array_100'], 
     all_ruptures['ruptures_other_100']]
)

ruptures_asp1_array_100 = filter_close_values(all_ruptures['ruptures_asp1_array_100'], 
                                                   [all_ruptures['ruptures_other_100']])

ruptures_asp2_array_100 = filter_close_values(all_ruptures['ruptures_asp2_array_100'], 
                                                   [all_ruptures['ruptures_other_100']])

#########################

values_list = list(all_ruptures.values())

len_long = len(values_list[2])
len_short = len(values_list[1])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['asp1', 'asp2', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]
colors = ['#c65102', 'olivedrab','cornflowerblue', 'plum']
# plot ruprure classification bar chart
plot_bar_chart(axs[5], categories, percentages, colors)

axs[5].set_xticks([])
axs[5].set_xticklabels([])
axs[5].set_ylabel('percentage (%)')
axs[5].set_xlabel('extra load 100 Pa asp1')
axs[5].legend(loc='upper right', fontsize=7.5, framealpha=0.7)

################### Recurrence time for central points of asperities
time_asp1 = values_list[1]
time_asp2 = values_list[2]
time_both = values_list[0]

time_asp1 = [t / 50 for t in time_asp1]
time_asp2 = [t / 50 for t in time_asp2]
time_both = [t / 50 for t in time_both]

time_diff_asp1 = [time_asp1[i] - time_asp1[i-1] for i in range(1, len(time_asp1))]
time_diff_asp2 = [time_asp2[i] - time_asp2[i-1] for i in range(1, len(time_asp2))]
time_diff_both = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[7].hist(time_diff_asp2, bins=bins, alpha=0.5, label='asp2', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[7].hist(time_diff_asp1, bins=bins, alpha=0.5, label='asp1', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[7].hist(time_diff_both, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[7].set_ylim(None, 30)
axs[7].set_xlim(None, 17)
axs[7].set_ylabel('frequency')
axs[7].legend()

###################### DOUBLE ASYMMETRIC ASP EXPS WITH THE SHORT ASP FIXED
# plot boxplot with CoV for double symmetric asp exps (axs[10])
rectangles = [((6, 0.1), 3, 0.7, '#D95319'), ((16, 0.1), 9, 0.7, 'olivedrab')]
plot_boxplot(axs[10], cv_matrix_40_40Pa_asymmetric_short_fixed, n_meas_points, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [0, 1.1])
# plot timeline map (axs[8])
x_ticks = np.arange(0, events_40_40Pa_asymmetric_short_fixed[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[8], events_40_40Pa_asymmetric_short_fixed[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')

##### boundaries
y2 = 24.65
y3 = 16  # upper boundary of long asp
y4 = 9   # lower boundary of short asp
y5 = 6   # upper boundary of short asp
axs[8].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[8].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)

axs[8].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[8].text(500, 17, 'long asp 40 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

### define events within asperities and outside
### rupture classification for double asymmetric exps
water = events_40_40Pa_asymmetric_short_fixed[:, 1500:]
threshold_main = 0.1

long_rows = np.arange(15, 25)
short_rows = np.arange(5, 9)
excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((short_rows, long_rows)))

ruptures_both_array_40 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and np.any(water[short_rows, j] > threshold_main)
]

ruptures_long_array_40 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and not np.any(water[short_rows, j] > threshold_main)
]

ruptures_short_array_40 = [
    j for j in range(water.shape[1]) 
    if np.any(water[short_rows, j] > threshold_main) and not np.any(water[long_rows, j] > threshold_main)
]

ruptures_other_40 = [
    j for j in range(water.shape[1]) 
    if not np.any(water[long_rows, j] > threshold_main) 
    and not np.any(water[short_rows, j] > threshold_main) 
    and np.any(water[excluded_rows, j] > threshold_main)
]

vector_names = {'ruptures_long_array_40': ruptures_long_array_40, 'ruptures_short_array_40': ruptures_short_array_40,
                'ruptures_both_array_40': ruptures_both_array_40, 'ruptures_other_40': ruptures_other_40}
result_vectors = process_vectors(vector_names)

ruptures_long_array_40 = result_vectors[0]
ruptures_short_array_40 = result_vectors[1]
ruptures_both_array_40 = result_vectors[2]
ruptures_other_40 = result_vectors[3]

all_ruptures = {
    'ruptures_both_array_40': ruptures_both_array_40,
    'ruptures_long_array_40': ruptures_long_array_40,
    'ruptures_short_array_40': ruptures_short_array_40,
    'ruptures_other_40': ruptures_other_40
}

ruptures_both_array_40 = filter_close_values(
    all_ruptures['ruptures_both_array_40'],
    [all_ruptures['ruptures_long_array_40'], all_ruptures['ruptures_short_array_40'], all_ruptures['ruptures_other_40']]
)

ruptures_long_array_40 = filter_close_values(all_ruptures['ruptures_long_array_40'], 
                                                   [all_ruptures['ruptures_other_40']])

ruptures_short_array_40 = filter_close_values(all_ruptures['ruptures_short_array_40'], 
                                                   [all_ruptures['ruptures_other_40']])

#########################
values_list = list(all_ruptures.values())

len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]

colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']
# plot ruprure classification bar chart
plot_bar_chart(axs[9], categories, percentages, colors)

axs[9].set_xticks([])
axs[9].set_xticklabels([])
axs[9].set_ylabel('percentage (%)')
axs[9].set_xlabel('extra load 40 Pa long asp')
axs[9].legend(loc='upper right', fontsize=7.5, framealpha=0.7)

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_40_40_short_fixed = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_40_40_short_fixed = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_40_40_short_fixed = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[11].hist(time_diff_long_40_40_short_fixed, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[11].hist(time_diff_short_40_40_short_fixed, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[11].hist(time_diff_both_40_40_short_fixed, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[11].set_ylim(None, 31)
axs[11].set_xlim(None, 17)
axs[11].set_ylabel('frequency')
axs[11].legend()

################# DOUBLE ASYMMETRIC ASP EXP 40-130 Pa WITH THE 40 PA SHORT ASP FIXED
# plot boxplot with CoV for double symmetric asp exps (axs[10])
rectangles = [((6, 0.1), 3, 0.7, '#D95319'), ((16, 0.1), 9, 0.7, 'olivedrab')]
plot_boxplot(axs[14], cv_matrix_40_130Pa_asymmetric_short_fixed, n_meas_points, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [0, 1.1])
# plot timeline map (axs[8]) (axs[12])
x_ticks = np.arange(0, events_40_130Pa_asymmetric_short_fixed[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[12], events_40_130Pa_asymmetric_short_fixed[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')

##### boundaries
y2 = 24.65
y3 = 16  # upper boundary of long asp
y4 = 9   # lower boundary of short asp
y5 = 6   # upper boundary of short asp
axs[12].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[12].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)

#axs[12].axvline(x=1600, color='goldenrod', linestyle='--', linewidth=1.5)
#axs[12].axvline(x=2020, color='goldenrod', linestyle='--', linewidth=1.5)



axs[12].text(500, 6, 'short asp fixed 40 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[12].text(500, 17, 'long asp 130 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
# rupture classification for 40_130Pa asymmetric short fixed
water = events_40_130Pa_asymmetric_short_fixed[:, 1500:]
threshold_main = 0.1

# define row ranges for long and short ruptures
long_rows = np.arange(15, 25)
short_rows = np.arange(5, 9)
excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((short_rows, long_rows)))

ruptures_both_array_130 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and np.any(water[short_rows, j] > threshold_main)
]

ruptures_long_array_130 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and not np.any(water[short_rows, j] > threshold_main)
]

ruptures_short_array_130 = [
    j for j in range(water.shape[1]) 
    if np.any(water[short_rows, j] > threshold_main) and not np.any(water[long_rows, j] > threshold_main)
]

ruptures_other_130 = [
    j for j in range(water.shape[1]) 
    if not np.any(water[long_rows, j] > threshold_main) 
    and not np.any(water[short_rows, j] > threshold_main) 
    and np.any(water[excluded_rows, j] > threshold_main)
]

    
vector_names = {'ruptures_long_array_130': ruptures_long_array_130, 'ruptures_short_array_130': ruptures_short_array_130,
                'ruptures_both_array_130': ruptures_both_array_130, 'ruptures_other_130': ruptures_other_130}
result_vectors = process_vectors(vector_names)

ruptures_long_array_130 = result_vectors[0]
ruptures_short_array_130 = result_vectors[1]
ruptures_both_array_130 = result_vectors[2]
ruptures_other_130 = result_vectors[3]

all_ruptures = {
    'ruptures_both_array_130': ruptures_both_array_130,
    'ruptures_long_array_130': ruptures_long_array_130,
    'ruptures_short_array_130': ruptures_short_array_130,
    'ruptures_other_130': ruptures_other_130
}

ruptures_both_array_130 = filter_close_values(
    all_ruptures['ruptures_both_array_130'],
    [all_ruptures['ruptures_long_array_130'], all_ruptures['ruptures_short_array_130'], all_ruptures['ruptures_other_130']]
)

ruptures_long_array_130 = filter_close_values(all_ruptures['ruptures_long_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])

ruptures_short_array_130 = filter_close_values(all_ruptures['ruptures_short_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])
##################

values_list = list(all_ruptures.values())
len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]
colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']
# plot ruprure classification bar chart
plot_bar_chart(axs[13], categories, percentages, colors)

axs[13].set_xticks([])
axs[13].set_xticklabels([])
axs[13].set_xlabel('extra load 130 Pa long asp')
axs[13].set_ylabel('percentage (%)')

################### Recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_130_40_short_fixed = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_130_40_short_fixed = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_130_40_short_fixed = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[15].hist(time_diff_long_130_40_short_fixed, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[15].hist(time_diff_short_130_40_short_fixed, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[15].hist(time_diff_both_130_40_short_fixed, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[15].set_ylim(None, 31)
axs[15].set_xlim(None, 17)
axs[15].set_ylabel('frequency')

################### DOUBLE ASYMMETRIC ASP EXP 40-70 Pa WITH THE 40 PA LONG ASP FIXED
# plot boxplot with CoV for double symmetric asp exps (axs[18])
# 40-70Pa asymmetric long fixed experiment (axs[18])
rectangles = [((6, 0.1), 3, 0.7, '#D95319'), ((16, 0.1), 9, 0.7, 'olivedrab')]
plot_boxplot(axs[18], cv_matrix_40_70Pa_asymmetric_long_fixed, n_meas_points, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [0, 1.1])
# plot timeline map (axs[16])
x_ticks = np.arange(0, events_40_70Pa_asymmetric_long_fixed[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[16], events_40_70Pa_asymmetric_long_fixed[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')

##### boundaries
y2 = 24.65
y3 = 16  # upper boundary of long asp
y4 = 9   # lower boundary of short asp
y5 = 6   # upper boundary of short asp
axs[16].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[16].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[16].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[16].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)


axs[16].text(500, 6, 'short asp 70 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[16].text(500, 17, 'long asp fixed 40 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
# rupture classification for 40_70Pa asymmetric long fixed
water = events_40_70Pa_asymmetric_long_fixed[:, 1500:]
threshold_main = 0.1

# define row ranges for long and short ruptures
long_rows = np.arange(15, 25)
short_rows = np.arange(5, 9)
excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((short_rows, long_rows)))

ruptures_both_array_40_70 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and np.any(water[short_rows, j] > threshold_main)
]

ruptures_long_array_40_70 = [
    j for j in range(water.shape[1]) 
    if np.any(water[long_rows, j] > threshold_main) and not np.any(water[short_rows, j] > threshold_main)
]

ruptures_short_array_40_70 = [
    j for j in range(water.shape[1]) 
    if np.any(water[short_rows, j] > threshold_main) and not np.any(water[long_rows, j] > threshold_main)
]

ruptures_other_40_70 = [
    j for j in range(water.shape[1]) 
    if not np.any(water[long_rows, j] > threshold_main) 
    and not np.any(water[short_rows, j] > threshold_main) 
    and np.any(water[excluded_rows, j] > threshold_main)
]

vector_names = {'ruptures_long_array_40_70': ruptures_long_array_40_70, 'ruptures_short_array_40_70': ruptures_short_array_40_70,
                'ruptures_both_array_40_70': ruptures_both_array_40_70, 'ruptures_other_40_70': ruptures_other_40_70}
result_vectors = process_vectors(vector_names)

ruptures_long_array_70 = result_vectors[0]
ruptures_short_array_70 = result_vectors[1]
ruptures_both_array_70 = result_vectors[2]
ruptures_other_70 = result_vectors[3]

all_ruptures = {
    'ruptures_both_array_70': ruptures_both_array_70,
    'ruptures_long_array_70': ruptures_long_array_70,
    'ruptures_short_array_70': ruptures_short_array_70,
    'ruptures_other_70': ruptures_other_70
}

ruptures_both_array_70 = filter_close_values(
    all_ruptures['ruptures_both_array_70'],
    [all_ruptures['ruptures_long_array_70'], all_ruptures['ruptures_short_array_70'], all_ruptures['ruptures_other_70']]
)

ruptures_long_array_70 = filter_close_values(all_ruptures['ruptures_long_array_70'], 
                                                   [all_ruptures['ruptures_other_70']])

ruptures_short_array_70 = filter_close_values(all_ruptures['ruptures_short_array_70'], 
                                                   [all_ruptures['ruptures_other_70']])

#########################
values_list = list(all_ruptures.values())
len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]
colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']
# plot rupture classification bar chart
plot_bar_chart(axs[17], categories, percentages, colors)
axs[17].set_xticks([])
axs[17].set_xticklabels([])
axs[17].set_ylabel('percentage (%)')
axs[17].set_xlabel('extra load 70 Pa short asp')

################### Recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_70_40_long_fixed = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_70_40_long_fixed = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_70_40_long_fixed = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[19].hist(time_diff_long_70_40_long_fixed, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[19].hist(time_diff_short_70_40_long_fixed, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[19].hist(time_diff_both_70_40_long_fixed, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[19].set_ylim(None, 31)
axs[19].set_xlim(None, 17)
axs[19].set_ylabel('frequency')

####################### DOUBLE ASYMMETRIC ASP EXP 40-130 Pa WITH THE 40 PA LONG ASP FIXED
# boxplot for 40-130Pa asymmetriclong asp fixed experiment (axs[22])
rectangles = [((6, 0.1), 3, 0.7, '#D95319'), ((16, 0.1), 9, 0.7, 'olivedrab')]
plot_boxplot(axs[22], cv_matrix_40_130Pa_asymmetric_long_fixed, n_meas_points, tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], rectangles, 'CoV', [0, 1.1])
axs[22].set_xlabel('distance along strike, cm')

# plot timeline map  for 40-130Pa asymmetriclong asp fixed experiment (axs[20])
x_ticks = np.arange(0, events_40_130Pa_asymmetric_long_fixed[:, 1500:].shape[1], 500)
x_labels = [f"{tick/50:.1f}" for tick in x_ticks]
plot_heatmap(axs[20], events_40_130Pa_asymmetric_long_fixed[:, 1500:], tick_positions, [f'{int(pos)}' for pos in tick_positions_cm], x_ticks, x_labels, 'events above threshold 0.1 cm/s', 'dist along strike, cm')
##### boundaries
y2 = 24.65
y3 = 16  # upper boundary of long asp
y4 = 9   # lower boundary of short asp
y5 = 6   # upper boundary of short asp
# transposing y3, y4, and y5 positions
axs[20].axhline(y=y2, color='#77AC30', linestyle='--', linewidth=1.5)
axs[20].axhline(y=y3, color='#77AC30', linestyle='--', linewidth=1.5)
axs[20].axhline(y=y4, color='#D95319', linestyle='--', linewidth=1.5)
axs[20].axhline(y=y5, color='#D95319', linestyle='--', linewidth=1.5)

axs[20].text(500, 6, 'short asp 130 Pa', color='#D95319', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)
axs[20].text(500, 17, 'long asp fixed 40 Pa', color='#77AC30', verticalalignment='bottom', horizontalalignment='center', fontsize=12)#, rotation=90)

axs[20].set_ylabel('dist along strike, cm')
axs[20].set_xlabel('time, seconds')

ruptures_long_array_130 = []
ruptures_short_array_130 = []
ruptures_both_array_130 = []
ruptures_other_130 = []

water = events_40_130Pa_asymmetric_long_fixed[:, 1500:]
for j in range(water.shape[1]):
    if np.any(water[15:25, j] > 0.1) and np.any(water[5:9, j] > 0.1):
        ruptures_both_array_130.append(j)
    elif np.any(water[15:25, j] > 0.1) and not np.any(water[5:9, j] > 0.1):
        ruptures_long_array_130.append(j)
    elif np.any(water[5:9, j] > 0.1) and not np.any(water[15:25, j] > 0.1):
        ruptures_short_array_130.append(j)
    else:
        # check if there's any value > 0.1 outside of rows 5-9 and 15-25
        excluded_rows = np.setdiff1d(np.arange(water.shape[0]), np.concatenate((np.arange(5, 9), np.arange(15, 25))))
        if np.any(water[excluded_rows, j] > 0.1):
            ruptures_other_130.append(j)
    
vector_names = {'ruptures_long_array_130': ruptures_long_array_130, 'ruptures_short_array_130': ruptures_short_array_130,
                'ruptures_both_array_130': ruptures_both_array_130, 'ruptures_other_130': ruptures_other_130}
result_vectors = process_vectors(vector_names)

ruptures_long_array_130 = result_vectors[0]
ruptures_short_array_130 = result_vectors[1]
ruptures_both_array_130 = result_vectors[2]
ruptures_other_130 = result_vectors[3]
##############################################
all_ruptures = {
    'ruptures_both_array_130': ruptures_both_array_130,
    'ruptures_long_array_130': ruptures_long_array_130,
    'ruptures_short_array_130': ruptures_short_array_130,
    'ruptures_other_130': ruptures_other_130
}

ruptures_both_array_130 = filter_close_values(
    all_ruptures['ruptures_both_array_130'],
    [all_ruptures['ruptures_long_array_130'], all_ruptures['ruptures_short_array_130'], all_ruptures['ruptures_other_130']]
)
ruptures_long_array_130 = filter_close_values(all_ruptures['ruptures_long_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])

ruptures_short_array_130 = filter_close_values(all_ruptures['ruptures_short_array_130'], 
                                                   [all_ruptures['ruptures_other_130']])

#########################
values_list = list(all_ruptures.values())
len_long = len(values_list[1])
len_short = len(values_list[2])
len_both = len(values_list[0])
len_other = len(values_list[3])

categories = ['short', 'long', 'both', 'other']
values = [len_short, len_long, len_both, len_other]
total_events = sum(values)
percentages = [v / total_events * 100 for v in values]
colors = ['#c65102', 'olivedrab', 'cornflowerblue', 'plum']
# plot rupture classification bar chart
plot_bar_chart(axs[21], categories, percentages, colors)

axs[21].set_xticks([])
axs[21].set_xticklabels([])
axs[21].set_xlabel('extra load 130 Pa short asp')
axs[21].set_ylabel('percentage (%)')

################### recurrence time for central points of asperities
time_long = values_list[1]
time_short = values_list[2]
time_both = values_list[0]

time_long = [t / 50 for t in time_long]
time_short = [t / 50 for t in time_short]
time_both = [t / 50 for t in time_both]

time_diff_long_130_40_long_fixed = [time_long[i] - time_long[i-1] for i in range(1, len(time_long))]
time_diff_short_130_40_long_fixed = [time_short[i] - time_short[i-1] for i in range(1, len(time_short))]
time_diff_both_130_40_long_fixed = [time_both[i] - time_both[i-1] for i in range(1, len(time_both))]

axs[23].hist(time_diff_long_130_40_long_fixed, bins=bins, alpha=0.5, label='long asp', color='olivedrab', edgecolor='gray', linewidth=0.5)
axs[23].hist(time_diff_short_130_40_long_fixed, bins=bins, alpha=0.5, label='short asp', color='darkorange', edgecolor='gray', linewidth=0.5)
axs[23].hist(time_diff_both_130_40_long_fixed, bins=bins, alpha=0.5, label='both asps', color='cornflowerblue', edgecolor='gray', linewidth=0.5)

axs[23].set_ylim(None, 31)
axs[23].set_xlim(None, 17)
axs[23].set_xlabel('time, seconds')
axs[23].set_ylabel('frequency')


fig.tight_layout(pad=3.0, h_pad=2.0, w_pad=2.0)
axs[7].sharex(axs[15])
axs[3].sharex(axs[7])
axs[11].sharex(axs[19])
axs[15].sharex(axs[11])
axs[23].sharex(axs[11])

plt.show()